﻿

namespace Hims.Api.Controllers
{
    using Hims.Api.Models;
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library.Enums;
    using Shared.UserModels;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/reference-type")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class ReferenceTypeController : BaseController
    {
        /// <summary>
        /// The ModulesMaster services.
        /// </summary>
        private readonly IReferenceTypesServices referenceTypeService;
        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public ReferenceTypeController(IReferenceTypesServices referenceTypeService, IAuditLogService auditLogService)
        {
            this.referenceTypeService = referenceTypeService;
            this.auditLogServices = auditLogService;
        }
      
        /// <summary>
        /// The fetch async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<ReferenceTypeModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] ReferenceTypeModel model, [FromHeader] LocationHeader location)
        {
            model = (ReferenceTypeModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var modules = await this.referenceTypeService.FetchReferences(model);         
            return this.Success(modules);

        }

        /// <summary>
        /// The create async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]        
        [Route("add")]
        public async Task<ActionResult> CreateAsync([FromBody] ReferenceTypeModel model, [FromHeader] LocationHeader location)
        {
            model = (ReferenceTypeModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var response = await this.referenceTypeService.AddReference(model);
        
            if (response > 0)
            {
                var auditLog = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.ReferralType,
                    LogDate = DateTime.Now,
                    LogFrom = short.Parse(model.LoginRoleId.ToString()),
                    LocationId = (int)model.LocationId,
                    LogDescription = $"<b>{model.CreatedByName}</b> has added <b>Referral Type </b> of <strong>{model.Name}</strong> successfully."
                };
                await this.auditLogServices.LogAsync(auditLog);
            }
        
        
        return this.Success(response);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="model"></param>
        /// <returns></returns>
        [HttpPost]
        [Route("update")]
        public async Task<ActionResult> UpdateAsync([FromBody] ReferenceTypeModel model, [FromHeader] LocationHeader location)
        {
            model = (ReferenceTypeModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var response = await this.referenceTypeService.UpdateReference(model);
            
                if (response > 0)
                {
                    var auditLog = new AuditLogModel
                    {
                        AccountId = model.CreatedBy,
                        LogTypeId = (int)LogTypes.ReferralType,
                        LogDate = DateTime.Now,
                        LogFrom = short.Parse(model.LoginRoleId.ToString()),
                        LocationId = (int)model.LocationId,
                        LogDescription = $"{model.CreatedByName} has updated <b>Referral Type</b> of <strong>{model.Name}</strong> successfully."
                    };
                    await this.auditLogServices.LogAsync(auditLog);
                }
            
            return this.Success(response);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="model"></param>
        /// <returns></returns>
        [HttpPost]        
        [Route("delete")]
        public async Task<ActionResult> DeleteAsync([FromBody] ReferenceTypeModel model, [FromHeader] LocationHeader location)
        {
            model = (ReferenceTypeModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var response = await this.referenceTypeService.DeleteReferences(model);
            if (response > 0)
            {
                var auditLog = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.ReferralType,
                    LogDate = DateTime.Now,
                    LogFrom = short.Parse(model.LoginRoleId.ToString()),
                    LocationId = (int)model.LocationId,
                    LogDescription = $"{model.CreatedByName} has deleted <b>Referral Type</b> of <strong>{model.Name}</strong> successfully."
                };
                await this.auditLogServices.LogAsync(auditLog);
            }

            return this.Success(response);
        }
    }
}
